<?php
/**
 * Settings class file.
 *
 * @package WordPress Plugin Template/Settings
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings class.
 */
class Supreme_Mega_Menu_Settings {

	/**
	 * The single instance of Supreme_Mega_Menu_Settings.
	 *
	 * @var     object
	 * @access  private
	 * @since   1.0.0
	 */
	private static $_instance = null; //phpcs:ignore

	/**
	 * The main plugin object.
	 *
	 * @var     object
	 * @access  public
	 * @since   1.0.0
	 */
	public $parent = null;

	/**
	 * Prefix for plugin settings.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $base = '';

	/**
	 * Available settings for plugin.
	 *
	 * @var     array
	 * @access  public
	 * @since   1.0.0
	 */
	public $settings = array();

	/**
	 * Constructor function.
	 *
	 * @param object $parent Parent object.
	 */
	public function __construct( $parent ) {
		$this->parent = $parent;

		$this->base = 'smega_menu_';

		// Initialise settings.
		add_action( 'init', array( $this, 'init_settings' ), 11 );

		// Register plugin settings.
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Add settings page to menu.
		add_action( 'admin_menu', array( $this, 'add_menu_item' ) );

		// Add settings link to plugins page.
		add_filter(
			'plugin_action_links_' . plugin_basename( $this->parent->file ),
			array(
				$this,
				'add_settings_link',
			)
		);

		// Configure placement of plugin settings page. See readme for implementation.
		add_filter( $this->base . 'menu_settings', array( $this, 'configure_settings' ) );
		// License page.
		global $smega_menu_license_sl;
		$smega_menu_license_sl = new SUPREME_MEGA_MENU_LICENSE_LOAD();
		add_action( 'init', array( $this, 'options_update' ), 1 );
		add_action( 'admin_notices', array( $this, 'admin_license_notices' ) );
		add_action( 'admin_notices', array( $this, 'admin_no_key_notices' ) );
		add_action( 'network_admin_notices', array( $this, 'admin_no_key_notices' ) );
		add_filter( $this->base . 'license_menu_settings', array( $this, 'configure_settings' ) );
		add_action( 'admin_menu', array( $this, 'license_add_menu_item' ) );
	}

	/**
	 * Initialise settings
	 *
	 * @return void
	 */
	public function init_settings() {
		$this->settings = $this->settings_fields();
	}

	/**
	 * Add settings page to admin menu
	 *
	 * @return void
	 */
	public function add_menu_item() {

		$args = $this->menu_settings();

		// Do nothing if wrong location key is set.
		if ( is_array( $args ) && isset( $args['location'] ) && function_exists( 'add_' . $args['location'] . '_page' ) ) {
			switch ( $args['location'] ) {
				case 'options':
				case 'submenu':
					$page = add_submenu_page( $args['parent_slug'], $args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function'] );
					break;
				case 'menu':
					$page = add_menu_page( $args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function'], $args['icon_url'], $args['position'] );
					break;
				default:
					return;
			}
			add_action( 'admin_print_styles-' . $page, array( $this, 'settings_assets' ) );
		}
	}

	/**
	 * Prepare default settings page arguments
	 *
	 * @return mixed|void
	 */
	private function menu_settings() {
		return apply_filters(
			$this->base . 'menu_settings',
			array(
				'location'    => 'menu', // Possible settings: options, menu, submenu.
				'parent_slug' => 'options-general.php',
				'page_title'  => __( 'Supreme Mega Menu', 'supreme-mega-menu' ),
				'menu_title'  => __( 'Supreme Mega Menu', 'supreme-mega-menu' ),
				'capability'  => 'manage_options',
				'menu_slug'   => $this->parent->_token . '_settings',
				'function'    => array( $this, 'settings_page' ),
				'icon_url'    => 'dashicons-menu-alt3',
				'position'    => null,
			)
		);
	}

	/**
	 * Container for settings page arguments
	 *
	 * @param array $settings Settings array.
	 *
	 * @return array
	 */
	public function configure_settings( $settings = array() ) {
		return $settings;
	}

	/**
	 * Load settings JS & CSS
	 *
	 * @return void
	 */
	public function settings_assets() {

		// We're including the farbtastic script & styles here because they're needed for the colour picker
		// If you're not including a colour picker field then you can leave these calls out as well as the farbtastic dependency for the wpt-admin-js script below.
		wp_enqueue_style( 'farbtastic' );
		wp_enqueue_script( 'farbtastic' );

		// We're including the WP media scripts here because they're needed for the image upload field.
		// If you're not including an image upload then you can leave this function call out.
		wp_enqueue_media();

		wp_register_script( $this->parent->_token . '-settings-js', $this->parent->assets_url . 'js/settings' . $this->parent->script_suffix . '.js', array( 'farbtastic', 'jquery' ), '1.0.0', true );
		wp_enqueue_script( $this->parent->_token . '-settings-js' );
	}

	/**
	 * Add settings link to plugin list table
	 *
	 * @param  array $links Existing links.
	 * @return array        Modified links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="options-general.php?page=' . $this->parent->_token . '_settings">' . __( 'Settings', 'supreme-mega-menu' ) . '</a>';
		array_push( $links, $settings_link );
		return $links;
	}

	/**
	 * Build settings fields
	 *
	 * @return array Fields to be displayed on settings page
	 */
	private function settings_fields() {
		/*
		$settings['standard'] = array(
			'title'       => __( 'Standard', 'supreme-mega-menu' ),
			'description' => __( 'These are fairly standard form input fields.', 'supreme-mega-menu' ),
			'fields'      => array(
				array(
					'id'          => 'text_field',
					'label'       => __( 'Some Text', 'supreme-mega-menu' ),
					'description' => __( 'This is a standard text field.', 'supreme-mega-menu' ),
					'type'        => 'text',
					'default'     => '',
					'placeholder' => __( 'Placeholder text', 'supreme-mega-menu' ),
				),
				array(
					'id'          => 'password_field',
					'label'       => __( 'A Password', 'supreme-mega-menu' ),
					'description' => __( 'This is a standard password field.', 'supreme-mega-menu' ),
					'type'        => 'password',
					'default'     => '',
					'placeholder' => __( 'Placeholder text', 'supreme-mega-menu' ),
				),
				array(
					'id'          => 'secret_text_field',
					'label'       => __( 'Some Secret Text', 'supreme-mega-menu' ),
					'description' => __( 'This is a secret text field - any data saved here will not be displayed after the page has reloaded, but it will be saved.', 'supreme-mega-menu' ),
					'type'        => 'text_secret',
					'default'     => '',
					'placeholder' => __( 'Placeholder text', 'supreme-mega-menu' ),
				),
				array(
					'id'          => 'text_block',
					'label'       => __( 'A Text Block', 'supreme-mega-menu' ),
					'description' => __( 'This is a standard text area.', 'supreme-mega-menu' ),
					'type'        => 'textarea',
					'default'     => '',
					'placeholder' => __( 'Placeholder text for this textarea', 'supreme-mega-menu' ),
				),
				array(
					'id'          => 'single_checkbox',
					'label'       => __( 'An Option', 'supreme-mega-menu' ),
					'description' => __( 'A standard checkbox - if you save this option as checked then it will store the option as \'on\', otherwise it will be an empty string.', 'supreme-mega-menu' ),
					'type'        => 'checkbox',
					'default'     => '',
				),
				array(
					'id'          => 'select_box',
					'label'       => __( 'A Select Box', 'supreme-mega-menu' ),
					'description' => __( 'A standard select box.', 'supreme-mega-menu' ),
					'type'        => 'select',
					'options'     => array(
						'drupal'    => 'Drupal',
						'joomla'    => 'Joomla',
						'WordPress' => 'WordPress',
					),
					'default'     => 'WordPress',
				),
				array(
					'id'          => 'radio_buttons',
					'label'       => __( 'Some Options', 'supreme-mega-menu' ),
					'description' => __( 'A standard set of radio buttons.', 'supreme-mega-menu' ),
					'type'        => 'radio',
					'options'     => array(
						'superman' => 'Superman',
						'batman'   => 'Batman',
						'ironman'  => 'Iron Man',
					),
					'default'     => 'batman',
				),
				array(
					'id'          => 'multiple_checkboxes',
					'label'       => __( 'Some Items', 'supreme-mega-menu' ),
					'description' => __( 'You can select multiple items and they will be stored as an array.', 'supreme-mega-menu' ),
					'type'        => 'checkbox_multi',
					'options'     => array(
						'square'    => 'Square',
						'circle'    => 'Circle',
						'rectangle' => 'Rectangle',
						'triangle'  => 'Triangle',
					),
					'default'     => array( 'circle', 'triangle' ),
				),
			),
		);

		$settings['extra'] = array(
			'title'       => __( 'Extra', 'supreme-mega-menu' ),
			'description' => __( 'These are some extra input fields that maybe aren\'t as common as the others.', 'supreme-mega-menu' ),
			'fields'      => array(
				array(
					'id'          => 'number_field',
					'label'       => __( 'A Number', 'supreme-mega-menu' ),
					'description' => __( 'This is a standard number field - if this field contains anything other than numbers then the form will not be submitted.', 'supreme-mega-menu' ),
					'type'        => 'number',
					'default'     => '',
					'placeholder' => __( '42', 'supreme-mega-menu' ),
				),
				array(
					'id'          => 'colour_picker',
					'label'       => __( 'Pick a colour', 'supreme-mega-menu' ),
					'description' => __( 'This uses WordPress\' built-in colour picker - the option is stored as the colour\'s hex code.', 'supreme-mega-menu' ),
					'type'        => 'color',
					'default'     => '#21759B',
				),
				array(
					'id'          => 'an_image',
					'label'       => __( 'An Image', 'supreme-mega-menu' ),
					'description' => __( 'This will upload an image to your media library and store the attachment ID in the option field. Once you have uploaded an imge the thumbnail will display above these buttons.', 'supreme-mega-menu' ),
					'type'        => 'image',
					'default'     => '',
					'placeholder' => '',
				),
				array(
					'id'          => 'multi_select_box',
					'label'       => __( 'A Multi-Select Box', 'supreme-mega-menu' ),
					'description' => __( 'A standard multi-select box - the saved data is stored as an array.', 'supreme-mega-menu' ),
					'type'        => 'select_multi',
					'options'     => array(
						'linux'   => 'Linux',
						'mac'     => 'Mac',
						'windows' => 'Windows',
					),
					'default'     => array( 'linux' ),
				),
			),
		);
		*/

		$settings['misc'] = array(
			'title'  => __( 'Misc', 'supreme-mega-menu' ),
			'fields' => array(
				array(
					'id'          => 'dynamic_css',
					'label'       => __( 'Enable Dynamic CSS', 'supreme-mega-menu' ),
					'description' => __(
						'Only load CSS files related to Supreme Mega Menu when they are needed on the page. This removes CSS files from the main bundle and improves load times. Dynamic CSS greatly reduces CSS file size by dynamically generating only the assets necessary for the features and modules you use. This eliminates all file bloat and greatly improves load times.<br>Note: Resaving the page is needed where the Supreme Mega Menu Module are being used to generate a new css file.',
						'supreme-mega-menu'
					),
					'type'        => 'checkbox',
				),
				array(
					'id'          => 'uninstall_on_delete',
					'label'       => __( 'Remove Data on Uninstall?', 'supreme-mega-menu' ),
					'description' => __( 'Check this box if you would like Supreme Mega Menu to completely remove all of its data when the plugin is deleted.', 'supreme-mega-menu' ),
					'type'        => 'checkbox',
				),
			),
		);

		$settings = apply_filters( $this->parent->_token . '_settings_fields', $settings );

		return $settings;
	}

	/**
	 * Register plugin settings
	 *
	 * @return void
	 */
	public function register_settings() {
		if ( is_array( $this->settings ) ) {

			// Check posted/selected tab.
			//phpcs:disable
			$current_section = '';
			if ( isset( $_POST['tab'] ) && $_POST['tab'] ) {
				$current_section = $_POST['tab'];
			} else {
				if ( isset( $_GET['tab'] ) && $_GET['tab'] ) {
					$current_section = $_GET['tab'];
				}
			}
			//phpcs:enable

			foreach ( $this->settings as $section => $data ) {

				if ( $current_section && $current_section !== $section ) {
					continue;
				}

				// Add section to page.
				add_settings_section( $section, $data['title'], array( $this, 'settings_section' ), $this->parent->_token . '_settings' );

				foreach ( $data['fields'] as $field ) {

					// Validation callback for field.
					$validation = '';
					if ( isset( $field['callback'] ) ) {
						$validation = $field['callback'];
					}

					// Register field.
					$option_name = $this->base . $field['id'];
					register_setting( $this->parent->_token . '_settings', $option_name, $validation );

					// Add field to page.
					add_settings_field(
						$field['id'],
						$field['label'],
						array( $this->parent->admin, 'display_field' ),
						$this->parent->_token . '_settings',
						$section,
						array(
							'field'  => $field,
							'prefix' => $this->base,
						)
					);
				}

				if ( ! $current_section ) {
					break;
				}
			}
		}
	}

	/**
	 * Settings section.
	 *
	 * @param array $section Array of section ids.
	 * @return void
	 */
	public function settings_section( $section ) {
		$html = '';
		if ( isset( $this->settings[ $section['id'] ]['description'] ) ) {
			$html = '<p> ' . $this->settings[ $section['id'] ]['description'] . '</p>' . "\n";
		}
		echo $html; //phpcs:ignore
	}

	/**
	 * Load settings page content.
	 *
	 * @return void
	 */
	public function settings_page() {

		// Build page HTML.
		$html      = '<div class="wrap" id="' . $this->parent->_token . '_settings">' . "\n";
			$html .= '<h2>' . __( 'Supreme Mega Menu Settings', 'supreme-mega-menu' ) . '</h2>' . "\n";

			$tab = '';
		//phpcs:disable
		if ( isset( $_GET['tab'] ) && $_GET['tab'] ) {
			$tab .= $_GET['tab'];
		}
		//phpcs:enable

		// Show page tabs.
		if ( is_array( $this->settings ) && 1 < count( $this->settings ) ) {

			$html .= '<h2 class="nav-tab-wrapper">' . "\n";

			$c = 0;
			foreach ( $this->settings as $section => $data ) {

				// Set tab class.
				$class = 'nav-tab';
				if ( ! isset( $_GET['tab'] ) ) { //phpcs:ignore
					if ( 0 === $c ) {
						$class .= ' nav-tab-active';
					}
				} else {
					if ( isset( $_GET['tab'] ) && $section == $_GET['tab'] ) { //phpcs:ignore
						$class .= ' nav-tab-active';
					}
				}

				// Set tab link.
				$tab_link = add_query_arg( array( 'tab' => $section ) );
				if ( isset( $_GET['settings-updated'] ) ) { //phpcs:ignore
					$tab_link = remove_query_arg( 'settings-updated', $tab_link );
				}

				// Output tab.
				$html .= '<a href="' . $tab_link . '" class="' . esc_attr( $class ) . '">' . esc_html( $data['title'] ) . '</a>' . "\n";

				++$c;
			}

			$html .= '</h2>' . "\n";
		}

			$html .= '<form method="post" action="options.php" enctype="multipart/form-data">' . "\n";

				// Get settings fields.
				ob_start();
				settings_fields( $this->parent->_token . '_settings' );
				do_settings_sections( $this->parent->_token . '_settings' );
				$html .= ob_get_clean();

				$html     .= '<p class="submit">' . "\n";
					$html .= '<input type="hidden" name="tab" value="' . esc_attr( $tab ) . '" />' . "\n";
					$html .= '<input name="Submit" type="submit" class="button-primary" value="' . esc_attr( __( 'Save Settings', 'supreme-mega-menu' ) ) . '" />' . "\n";
				$html     .= '</p>' . "\n";
			$html         .= '</form>' . "\n";
		$html             .= '</div>' . "\n";

		echo $html; //phpcs:ignore
	}

	/**
	 * Main Supreme_Mega_Menu_Settings Instance
	 *
	 * Ensures only one instance of Supreme_Mega_Menu_Settings is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @see Supreme_Mega_Menu()
	 * @param object $parent Object instance.
	 * @return object Supreme_Mega_Menu_Settings instance
	 */
	public static function instance( $parent ) {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self( $parent );
		}
		return self::$_instance;
	} // End instance()

	/**
	 * Cloning is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html( __( 'Cloning of Supreme_Mega_Menu_API is forbidden.' ) ), esc_attr( $this->parent->_version ) );
	} // End __clone()

	/**
	 * Unserializing instances of this class is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html( __( 'Unserializing instances of Supreme_Mega_Menu_API is forbidden.' ) ), esc_attr( $this->parent->_version ) );
	} // End __wakeup()

	// License.
		/**
		 * Prepare default settings page arguments
		 *
		 * @return mixed|void
		 */
	private function license_menu_settings() {
		return apply_filters(
			$this->base . 'license_menu_settings',
			array(
				'location'    => 'submenu', // Possible settings: options, menu, submenu.
				'parent_slug' => 'supreme_mega_menu_settings',
				'page_title'  => __( 'License', 'supreme-mega-menu' ),
				'menu_title'  => __( 'License', 'supreme-mega-menu' ),
				'capability'  => 'manage_options',
				'menu_slug'   => $this->parent->_token . '_license_settings',
				'function'    => array( $this, 'plugin_options_page' ),
				'icon_url'    => '',
				'position'    => null,
			)
		);
	}
		/**
		 * Add settings page to admin menu
		 *
		 * @return void
		 */
	public function license_add_menu_item() {

		$args = $this->license_menu_settings();

		// Do nothing if wrong location key is set.
		if ( is_array( $args ) && isset( $args['location'] ) && function_exists( 'add_' . $args['location'] . '_page' ) ) {
			switch ( $args['location'] ) {
				case 'options':
				case 'submenu':
					$page = add_submenu_page( $args['parent_slug'], $args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function'] );
					break;
				case 'menu':
					$page = add_menu_page( $args['page_title'], $args['menu_title'], $args['capability'], $args['menu_slug'], $args['function'], $args['icon_url'], $args['position'] );
					break;
				default:
					return;
			}
			add_action( 'admin_print_styles-' . $page, array( $this, 'settings_assets' ) );
		}
	}

		/**
		 * License admin_no_key_notices
		 */
	public function admin_no_key_notices() {
		global $smega_menu_license_sl;

		if ( $smega_menu_license_sl->licence->licence_key_verify() === true ) {
			return;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$screen = get_current_screen();

		if ( ! Supreme_MegaMenu_NOTICE::is_admin_notice_active( 'disable-activate-license-notice-forever' ) ) {
			return;
		}

		if ( is_multisite() ) {
			?><div data-dismissible="disable-install-lite-notice-forever" class="notice notice-warning is-dismissible"><p><?php esc_html_e( 'Supreme Mega Menu: please enter your', 'supreme-mega-menu' ); ?> <a href="<?php echo esc_url( network_admin_url() ); ?>settings.php?page=supreme_mega_menu_license_settings"><?php esc_html_e( 'Licence Key', 'supreme-mega-menu' ); ?></a> to get regular updates.</p></div>
				<?php
		} else {
			?>
					<div data-dismissible="disable-install-lite-notice-forever" class="notice notice-warning is-dismissible"><p><?php esc_html_e( 'Supreme Mega Menu: please enter your', 'supreme-mega-menu' ); ?> <a href="admin.php?page=supreme_mega_menu_license_settings"><?php esc_html_e( 'Licence Key', 'supreme-mega-menu' ); ?></a> to get regular updates.</p></div>
				<?php
		}
	}

		/**
		 * License all_admin_notices
		 */
	public function admin_license_notices() {
		global $smega_menu_interface_messages;

		if ( ! is_array( $smega_menu_interface_messages ) ) {
			return;
		}

		if ( count( $smega_menu_interface_messages ) > 0 ) {
			foreach ( $smega_menu_interface_messages  as  $message ) {
				echo "<div class='" . esc_attr( $message['type'] ) . " notice-warning notice fade'><p>" . esc_html( $message['text'] ) . '</p></div>';
			}
		}

	}

		/**
		 * On network plugin options interface save
		 */
	public function network_options_update() {
		if ( isset( $_POST['smega_menu_licence_form_submit'] ) && isset( $_POST['smega_menu_license_key'] ) && wp_verify_nonce( sanitize_key( $_POST['smega_menu_license_key'] ), 'smega_menu_license_key' ) ) {
			$this->licence_form_submit();
		}

	}

		/**
		 * On options save
		 */
	public function options_update() {
		if ( isset( $_POST['smega_menu_licence_form_submit'] ) && isset( $_POST['smega_menu_license_key'] ) && wp_verify_nonce( sanitize_key( $_POST['smega_menu_license_key'] ), 'smega_menu_license_key' ) ) {
			$this->licence_form_submit();
		}
	}

		/**
		 *
		 * The license plugin options page
		 */
	public function plugin_options_page() {
		global $smega_menu_license_sl;

		if ( ! $smega_menu_license_sl->licence->licence_key_verify() ) {
			$this->licence_form();
			return;
		}

		if ( $smega_menu_license_sl->licence->licence_key_verify() ) {
			$this->licence_deactivate_form();
		}

	}

		/**
		 *
		 * Licence form submit
		 */
	public function licence_form_submit() {
		global $smega_menu_interface_messages;

		// check for de-activation.
		if ( isset( $_POST['smega_menu_licence_form_submit'] ) && isset( $_POST['smega_menu_licence_deactivate'] ) && isset( $_POST['smega_menu_license_key'] ) && wp_verify_nonce( sanitize_key( $_POST['smega_menu_license_key'] ), 'smega_menu_license_key' ) ) {
			$licence_data = get_site_option( 'smega_menu_license_key' );
			$licence_key  = $licence_data['key'];

			// build the request query.
			$args        = array(
				'woo_sl_action'     => 'deactivate',
				'licence_key'       => $licence_key,
				'product_unique_id' => SMEGA_MENU_PRODUCT_ID,
				'domain'            => SMEGA_MENU_INSTANCE,
			);
			$request_uri = SMEGA_MENU_APP_API_URL . '?' . http_build_query( $args, '', '&' );
			$data        = wp_safe_remote_get( $request_uri );

			if ( is_wp_error( $data ) || 200 !== $data['response']['code'] ) {
					$smega_menu_interface_messages[] = array(
						'type' => 'error',
						'text' => __( 'There was a problem connecting to ', 'supreme-mega-menu' ) . SMEGA_MENU_APP_API_URL,
					);
					return;
			}

			$response_block = json_decode( $data['body'] );
			$response_block = $response_block[ count( $response_block ) - 1 ];
			$response       = $response_block->message;

			if ( isset( $response_block->status ) ) {
				if ( 'success' === $response_block->status && 's201' === $response_block->status_code ) {
							// the license is active and the software is active.
							$smega_menu_interface_messages[] = array(
								'type' => 'updated',
								'text' => $response_block->message,
							);

							$licence_data = get_site_option( 'smega_menu_license_key' );

							// save the license.
							$licence_data['key']        = '';
							$licence_data['last_check'] = time();

							update_site_option( 'smega_menu_license_key', $licence_data );
				} else { // if message code is e104  force de-activation.
					if ( 'e002' === $response_block->status_code || 'e104' === $response_block->status_code || 'e211' === $response_block->status_code ) {
								$licence_data = get_site_option( 'smega_menu_license_key' );

								// save the license.
								$licence_data['key']        = '';
								$licence_data['last_check'] = time();

								update_site_option( 'smega_menu_license_key', $licence_data );
					} else {
						$smega_menu_interface_messages[] = array(
							'type' => 'error',
							'text' => __( 'There was a problem deactivating the licence: ', 'supreme-mega-menu' ) . $response_block->message,
						);

						return;
					}
				}
			} else {
						$smega_menu_interface_messages[] = array(
							'type' => 'error',
							'text' => __( 'There was a problem with the data block received from ', 'supreme-mega-menu' ) . SMEGA_MENU_APP_API_URL,
						);
						return;
			}
		}

		if ( isset( $_POST['smega_menu_licence_form_submit'] ) && isset( $_POST['smega_menu_licence_activate'] ) && isset( $_POST['smega_menu_license_key'] ) && wp_verify_nonce( sanitize_key( $_POST['smega_menu_license_key'] ), 'smega_menu_license_key' ) ) {
			$licence_key = isset( $_POST['licence_key'] ) ? trim( sanitize_key( $_POST['licence_key'] ) ) : '';

			if ( '' === $licence_key ) {
				$smega_menu_interface_messages[] = array(
					'type' => 'error',
					'text' => __( "Licence Key can't be empty", 'supreme-mega-menu' ),
				);
				return;
			}

			// build the request query.
			$args                    = array(
				'woo_sl_action'     => 'activate',
				'licence_key'       => $licence_key,
				'product_unique_id' => SMEGA_MENU_PRODUCT_ID,
				'domain'            => SMEGA_MENU_INSTANCE,
			);
						$request_uri = SMEGA_MENU_APP_API_URL . '?' . http_build_query( $args, '', '&' );
						$data        = wp_safe_remote_get( $request_uri );

			if ( is_wp_error( $data ) || 200 !== $data['response']['code'] ) {
				$smega_menu_interface_messages[] = array(
					'type' => 'error',
					'text' => __( 'There was a problem connecting to ', 'supreme-mega-menu' ) . SMEGA_MENU_APP_API_URL,
				);
				return;
			}

						$response_block = json_decode( $data['body'] );
						// retrieve the last message within the $response_block.
						$response_block = $response_block[ count( $response_block ) - 1 ];
						$response       = $response_block->message;

			if ( isset( $response_block->status ) ) {
				if ( 'success' === $response_block->status && ( 's100' === $response_block->status_code || 's101' === $response_block->status_code ) ) {
						// the license is active and the software is active.
						$smega_menu_interface_messages[] = array(
							'type' => 'updated',
							'text' => $response_block->message,
						);

						$licence_data = get_site_option( 'smega_menu_license_key' );

						// save the license.
						$licence_data['key']        = $licence_key;
						$licence_data['last_check'] = time();

						update_site_option( 'smega_menu_license_key', $licence_data );

				} else {
					$smega_menu_interface_messages[] = array(
						'type' => 'error',
						'text' => __( 'There was a problem activating the licence: ', 'supreme-mega-menu' ) . $response_block->message,
					);
							return;
				}
			} else {
				$smega_menu_interface_messages[] = array(
					'type' => 'error',
					'text' => __( 'There was a problem with the data block received from ', 'supreme-mega-menu' ) . SMEGA_MENU_APP_API_URL,
				);
				return;
			}
		}
	}

		/**
		 *
		 * Licence form
		 */
	public function licence_form() {
		?>
				<div class="wrap"> 
					<h2><?php esc_html_e( 'Supreme Mega Menu License', 'supreme-mega-menu' ); ?></h2>
						<form id="form_data" name="form" method="post">
						<?php wp_nonce_field( 'smega_menu_license_key', 'smega_menu_license_key' ); ?>
							<input type="hidden" name="smega_menu_licence_form_submit" value="true" />
							<input type="hidden" name="smega_menu_licence_activate" value="true" />
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th>License Key</th>
										<td>
											<input type="text" value="" name="licence_key" class="text-input">
											<label for="text_field">
												<span class="description"><?php esc_html_e( 'Enter the Licence Key you received when purchased this product. If you lost the key, you can always retrieve it from', 'supreme-mega-menu' ); ?> <a href="https://divisupreme.com/my-account/" target="_blank"><?php esc_html_e( 'My Account', 'supreme-mega-menu' ); ?></a></span>
											</label>
										</td>
									</tr>
								</tbody>
							</table>
							<p class="submit">
								<input type="submit" name="Submit" class="button-primary" value="<?php esc_html_e( 'Save & Activate License Key', 'supreme-mega-menu' ); ?>">
							</p>
						</form> 
				</div> 
			<?php
	}

		/**
		 *
		 * Licence deactivate form
		 */
	public function licence_deactivate_form() {
			global $smega_menu_license_sl;
			$licence_data = get_site_option( 'smega_menu_license_key' );

		?>
				<div class="wrap"> 
					<h2><?php esc_html_e( 'Supreme Menu Menu', 'supreme-mega-menu' ); ?></h2>
					<br />
					<div class="start-container">
						<div id="form_data">
								<form id="form_data" name="form" method="post">    
							<?php wp_nonce_field( 'smega_menu_license_key', 'smega_menu_license_key' ); ?>
									<input type="hidden" name="smega_menu_licence_form_submit" value="true" />
									<input type="hidden" name="smega_menu_licence_deactivate" value="true" />
									<table class="form-table" role="presentation">
										<tbody>
											<tr>
												<th>License Key</th>
												<td>
											<?php
											if ( $smega_menu_license_sl->licence->is_local_instance() ) {
												?>
															<p><?php esc_html_e( 'Local instance, no key applied.', 'supreme-mega-menu' ); ?></p>
													<?php
											} else {
												?>
															<p><b><?php echo esc_attr( substr( $licence_data['key'], 0, 20 ) ); ?>-xxxxxxxx-xxxxxxxx</b> &nbsp;&nbsp;&nbsp;</p>
													<?php } ?>
													<label for="text_field">
														<span class="description"><?php esc_html_e( 'You can generate more keys from', 'supreme-mega-menu' ); ?> <a href="https://divisupreme.com/my-account/" target="_blank"><?php esc_html_e( 'My Account', 'supreme-mega-menu' ); ?></a></span>
													</label>
												</td>
											</tr> 
										</tbody>
									</table>
									<p class="submit">
										<input type="submit" name="Submit" class="button-secondary"  title="Deactivate" href="javascript: void(0)" onclick="jQuery(this).closest('form').submit();" value="<?php esc_html_e( 'Deactivate', 'supreme-mega-menu' ); ?>">
									</p>
								</form>

						</div>
					</div>
				</div> 
			<?php

	}
}
